<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Admin;
use App\Models\Gift;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\PersonalAccessToken;

class GiftController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Credit Gift to User
     * Admin can send a gift (amount) to a user
     */
    public function creditGift(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:0.01',
            'message' => 'nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::find($request->user_id);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        // Use database transaction to ensure atomicity
        DB::transaction(function () use ($user, $request, $admin) {
            // Credit amount to user's wallet
            $user->wallet_balance += $request->amount;
            $user->save();

            // Create gift record
            Gift::create([
                'user_id' => $user->id,
                'amount' => $request->amount,
                'message' => $request->message ?? 'You received a gift!',
                'is_read' => false,
                'created_by' => $admin->id,
            ]);
        });

        return response()->json([
            'success' => true,
            'message' => 'Gift credited successfully',
            'data' => [
                'user_id' => $user->id,
                'amount' => (float) $request->amount,
                'message' => $request->message ?? 'You received a gift!',
                'new_wallet_balance' => (float) $user->fresh()->wallet_balance,
            ]
        ], 200);
    }

    /**
     * Get All Gifts (Admin View)
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $query = Gift::with(['user:id,email,phone_number,name', 'admin:id,username']);

        // Filter by user_id
        if ($request->has('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by is_read
        if ($request->has('is_read')) {
            $query->where('is_read', $request->is_read === 'true' || $request->is_read === '1');
        }

        // Filter by date range
        if ($request->has('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }

        // Sort options
        $sortBy = $request->input('sort_by', 'created_at');
        $sortOrder = $request->input('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100);

        $gifts = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $gifts->items(),
            'pagination' => [
                'current_page' => $gifts->currentPage(),
                'per_page' => $gifts->perPage(),
                'total' => $gifts->total(),
                'last_page' => $gifts->lastPage(),
            ]
        ], 200);
    }
}

