<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\RechargeOffer;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class RechargeOfferController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get All Recharge Offers
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $offers = RechargeOffer::orderBy('created_at', 'desc')->get();

        return response()->json([
            'success' => true,
            'data' => $offers
        ], 200);
    }

    /**
     * Get Single Recharge Offer
     */
    public function show(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $offer = RechargeOffer::find($id);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Recharge offer not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $offer
        ], 200);
    }

    /**
     * Add New Recharge Offer
     */
    public function store(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'minimum_recharge_amount' => 'required|numeric|min:0.01',
            'discount_type' => 'nullable|in:percentage,fixed',
            'discount_value' => 'nullable|numeric|min:0|required_with:discount_type',
            'cashback_amount' => 'nullable|numeric|min:0',
            'validity_start_date' => 'required|date',
            'validity_end_date' => 'required|date|after_or_equal:validity_start_date',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $offer = RechargeOffer::create([
            'title' => $request->title,
            'description' => $request->description,
            'minimum_recharge_amount' => $request->minimum_recharge_amount,
            'discount_type' => $request->discount_type,
            'discount_value' => $request->discount_value,
            'cashback_amount' => $request->cashback_amount ?? 0.00,
            'validity_start_date' => $request->validity_start_date,
            'validity_end_date' => $request->validity_end_date,
            'is_active' => $request->is_active ?? true,
            'created_by' => $admin->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Recharge offer added successfully',
            'data' => $offer
        ], 201);
    }

    /**
     * Update Recharge Offer
     */
    public function update(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'minimum_recharge_amount' => 'sometimes|required|numeric|min:0.01',
            'discount_type' => 'nullable|in:percentage,fixed',
            'discount_value' => 'nullable|numeric|min:0|required_with:discount_type',
            'cashback_amount' => 'nullable|numeric|min:0',
            'validity_start_date' => 'sometimes|required|date',
            'validity_end_date' => 'sometimes|required|date|after_or_equal:validity_start_date',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $offer = RechargeOffer::find($id);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Recharge offer not found'
            ], 404);
        }

        $offer->update($request->only([
            'title', 'description', 'minimum_recharge_amount', 
            'discount_type', 'discount_value', 'cashback_amount',
            'validity_start_date', 'validity_end_date', 'is_active'
        ]));

        return response()->json([
            'success' => true,
            'message' => 'Recharge offer updated successfully',
            'data' => $offer->fresh()
        ], 200);
    }

    /**
     * Delete Recharge Offer
     */
    public function destroy(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $offer = RechargeOffer::find($id);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Recharge offer not found'
            ], 404);
        }

        $offer->delete();

        return response()->json([
            'success' => true,
            'message' => 'Recharge offer deleted successfully'
        ], 200);
    }
}
