<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Slider;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class SliderController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get All Sliders
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $sliders = Slider::with('admin')->orderBy('order', 'asc')->orderBy('created_at', 'desc')->get();

        return response()->json([
            'success' => true,
            'data' => $sliders
        ], 200);
    }

    /**
     * Get Single Slider
     */
    public function show(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $slider = Slider::with('admin')->find($id);

        if (!$slider) {
            return response()->json([
                'success' => false,
                'message' => 'Slider not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $slider
        ], 200);
    }

    /**
     * Add New Slider
     */
    public function store(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'image_url' => 'required|string|url',
            'link' => 'nullable|string|url',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $slider = Slider::create([
            'title' => $request->title,
            'description' => $request->description,
            'image_url' => $request->image_url,
            'link' => $request->link,
            'order' => $request->order ?? 0,
            'is_active' => $request->is_active ?? true,
            'created_by' => $admin->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Slider added successfully',
            'data' => $slider->load('admin')
        ], 201);
    }

    /**
     * Update Slider
     */
    public function update(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'image_url' => 'sometimes|required|string|url',
            'link' => 'nullable|string|url',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $slider = Slider::find($id);

        if (!$slider) {
            return response()->json([
                'success' => false,
                'message' => 'Slider not found'
            ], 404);
        }

        $slider->update($request->only(['title', 'description', 'image_url', 'link', 'order', 'is_active']));

        return response()->json([
            'success' => true,
            'message' => 'Slider updated successfully',
            'data' => $slider->fresh()->load('admin')
        ], 200);
    }

    /**
     * Delete Slider
     */
    public function destroy(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $slider = Slider::find($id);

        if (!$slider) {
            return response()->json([
                'success' => false,
                'message' => 'Slider not found'
            ], 404);
        }

        $slider->delete();

        return response()->json([
            'success' => true,
            'message' => 'Slider deleted successfully'
        ], 200);
    }
}
