<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Service;
use App\Models\ServiceTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\PersonalAccessToken;

class FarmerIdController extends Controller
{
    /**
     * Get authenticated user from token
     */
    private function getUser(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $user = $accessToken->tokenable;
        if (!$user || !($user instanceof User)) {
            return null;
        }

        return $user;
    }

    /**
     * Verify Farmer ID using Aadhar Number
     */
    public function verify(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'aadhar' => 'required|string|size:12|regex:/^[0-9]{12}$/',
            'state' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // Get Farmer ID service
        $service = Service::where('path', '/user/farmerid')->first();
        
        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Farmer ID service not found. Please contact admin.'
            ], 404);
        }

        if ($service->is_coming_soon) {
            return response()->json([
                'success' => false,
                'message' => 'This service is coming soon.'
            ], 400);
        }

        $serviceCost = $service->cost ?? 25.00; // Default to ₹25 if not set

        // Check wallet balance
        if ($user->wallet_balance < $serviceCost) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient wallet balance',
                'data' => [
                    'required_amount' => $serviceCost,
                    'current_balance' => $user->wallet_balance,
                    'shortage' => $serviceCost - $user->wallet_balance
                ]
            ], 400);
        }

        // Create transaction record
        $transaction = ServiceTransaction::create([
            'user_id' => $user->id,
            'service_id' => $service->id,
            'amount' => $serviceCost,
            'service_type' => 'farmer_id',
            'request_data' => ['aadhar' => $request->aadhar, 'state' => $request->state],
            'status' => 'pending',
        ]);

        try {
            // Call external API (timeout set to 3 minutes to handle 1-2 minute response time)
            $response = Http::timeout(180)->get('http://alldigitalservice.cloud/verify_card_details', [
                'state_name' => $request->state,
                'aadhar' => $request->aadhar
            ]);

            $responseData = $response->json();

            // Check if API call was successful
            if ($response->successful() && isset($responseData['status']) && $responseData['status'] == 200) {
                // Deduct amount from wallet
                DB::transaction(function () use ($user, $serviceCost, $transaction, $responseData) {
                    $user->wallet_balance -= $serviceCost;
                    $user->save();

                    // Update transaction
                    $transaction->update([
                        'status' => 'success',
                        'response_data' => $responseData,
                    ]);
                });

                // Return farmer data
                return response()->json([
                    'success' => true,
                    'message' => 'Farmer ID verified successfully',
                    'data' => [
                        'transaction_id' => $transaction->id,
                        'farmer_id' => $responseData['data']['farmerID'] ?? null,
                        'farmer_aadhar_number' => $responseData['data']['farmerAadharNumber'] ?? null,
                        'farmer_dob' => $responseData['data']['farmerDob'] ?? null,
                        'farmer_name_en' => $responseData['data']['farmerNameEn'] ?? null,
                        'farmer_name_local' => $responseData['data']['farmerNameLocal'] ?? null,
                        'farmer_gender' => $responseData['data']['farmerGenederType'] ?? null,
                        'farmer_pincode' => $responseData['data']['farmerPinCode'] ?? null,
                        'farmer_address' => $responseData['data']['farmerAddressEn'] ?? null,
                        'farmer_mobile' => $responseData['data']['farmerMobileNumber'] ?? null,
                        'farmer_photograph' => $responseData['data']['ferFarmerPhotograph'] ?? null,
                        'farmer_land_ownerships' => $responseData['data']['farmerLandOwnerShips'] ?? [],
                        'amount_deducted' => $serviceCost,
                        'remaining_balance' => $user->fresh()->wallet_balance,
                    ]
                ], 200);
            } else {
                // API call failed or returned error
                $transaction->update([
                    'status' => 'failed',
                    'response_data' => $responseData,
                    'error_message' => $responseData['message'] ?? 'API call failed',
                ]);

                return response()->json([
                    'success' => false,
                    'message' => $responseData['message'] ?? 'Failed to verify Farmer ID. Please try again.',
                    'data' => $responseData
                ], 400);
            }
        } catch (\Exception $e) {
            // Handle exceptions
            $transaction->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while verifying Farmer ID. Please try again later.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get Transaction Status by ID
     */
    public function getStatus(Request $request, $id)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $transaction = ServiceTransaction::where('id', $id)
            ->where('user_id', $user->id)
            ->with(['service:id,name,path'])
            ->first();

        if (!$transaction) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction not found'
            ], 404);
        }

        $responseData = [
            'id' => $transaction->id,
            'status' => $transaction->status,
            'amount' => $transaction->amount,
            'service' => $transaction->service,
            'created_at' => $transaction->created_at,
            'updated_at' => $transaction->updated_at,
        ];

        // If successful, include farmer data
        if ($transaction->status === 'success' && $transaction->response_data) {
            $farmerData = $transaction->response_data['data'] ?? null;
            if ($farmerData) {
                $responseData['farmer_data'] = [
                    'farmer_id' => $farmerData['farmerID'] ?? null,
                    'farmer_aadhar_number' => $farmerData['farmerAadharNumber'] ?? null,
                    'farmer_dob' => $farmerData['farmerDob'] ?? null,
                    'farmer_name_en' => $farmerData['farmerNameEn'] ?? null,
                    'farmer_name_local' => $farmerData['farmerNameLocal'] ?? null,
                    'farmer_gender' => $farmerData['farmerGenederType'] ?? null,
                    'farmer_pincode' => $farmerData['farmerPinCode'] ?? null,
                    'farmer_address' => $farmerData['farmerAddressEn'] ?? null,
                    'farmer_mobile' => $farmerData['farmerMobileNumber'] ?? null,
                    'farmer_photograph' => $farmerData['ferFarmerPhotograph'] ?? null,
                    'farmer_land_ownerships' => $farmerData['farmerLandOwnerShips'] ?? [],
                ];
            }
        }

        // If failed, include error message
        if ($transaction->status === 'failed') {
            $responseData['error_message'] = $transaction->error_message;
        }

        return response()->json([
            'success' => true,
            'data' => $responseData
        ], 200);
    }

    /**
     * Get Service Transaction History
     */
    public function history(Request $request)
    {
        $user = $this->getUser($request);
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Please login first.'
            ], 401);
        }

        $transactions = ServiceTransaction::where('user_id', $user->id)
            ->with(['service:id,name,path'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $transactions
        ], 200);
    }
}
