<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\RechargeOffer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class RechargeOfferController extends Controller
{
    /**
     * Get All Active Recharge Offers
     * Returns only valid and active offers
     */
    public function index(Request $request)
    {
        $today = now()->toDateString();
        
        $offers = RechargeOffer::where('is_active', true)
            ->where('validity_start_date', '<=', $today)
            ->where('validity_end_date', '>=', $today)
            ->orderBy('minimum_recharge_amount', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $offers
        ], 200);
    }

    /**
     * Get Single Recharge Offer
     */
    public function show($id)
    {
        $offer = RechargeOffer::find($id);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Recharge offer not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $offer,
            'is_valid' => $offer->isValid()
        ], 200);
    }

    /**
     * Get Offers for Specific Recharge Amount
     * Returns offers applicable for the given recharge amount
     */
    public function getApplicableOffers(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'recharge_amount' => 'required|numeric|min:0.01',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $rechargeAmount = $request->recharge_amount;
        $today = now()->toDateString();

        $offers = RechargeOffer::where('is_active', true)
            ->where('validity_start_date', '<=', $today)
            ->where('validity_end_date', '>=', $today)
            ->where('minimum_recharge_amount', '<=', $rechargeAmount)
            ->orderBy('minimum_recharge_amount', 'desc')
            ->get();

        // Calculate benefits for each offer
        $offersWithBenefits = $offers->map(function($offer) use ($rechargeAmount) {
            $benefit = 0;
            $benefitDescription = '';

            if ($offer->discount_type === 'percentage' && $offer->discount_value) {
                $benefit = ($rechargeAmount * $offer->discount_value) / 100;
                $benefitDescription = "{$offer->discount_value}% discount = ₹{$benefit}";
            } elseif ($offer->discount_type === 'fixed' && $offer->discount_value) {
                $benefit = $offer->discount_value;
                $benefitDescription = "₹{$benefit} discount";
            }

            if ($offer->cashback_amount > 0) {
                $benefit += $offer->cashback_amount;
                $benefitDescription .= $benefitDescription ? " + ₹{$offer->cashback_amount} cashback" : "₹{$offer->cashback_amount} cashback";
            }

            return [
                'id' => $offer->id,
                'title' => $offer->title,
                'description' => $offer->description,
                'minimum_recharge_amount' => $offer->minimum_recharge_amount,
                'discount_type' => $offer->discount_type,
                'discount_value' => $offer->discount_value,
                'cashback_amount' => $offer->cashback_amount,
                'total_benefit' => $benefit,
                'benefit_description' => $benefitDescription,
                'validity_start_date' => $offer->validity_start_date,
                'validity_end_date' => $offer->validity_end_date,
            ];
        });

        return response()->json([
            'success' => true,
            'recharge_amount' => $rechargeAmount,
            'data' => $offersWithBenefits
        ], 200);
    }
}
