<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class ServiceController extends Controller
{
    /**
     * Check Service Status by Path
     * Used by Angular frontend to check if service is available or coming soon
     */
    public function checkByPath(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'path' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $service = Service::where('path', $request->path)->first();

        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Service not found',
                'is_coming_soon' => false,
                'available' => false
            ], 404);
        }

        $serviceData = [
            'id' => $service->id,
            'name' => $service->name,
            'path' => $service->path,
            'is_coming_soon' => $service->is_coming_soon,
            'available' => !$service->is_coming_soon,
            'message' => $service->is_coming_soon ? 'This service will come soon' : 'Service is available'
        ];

        if ($service->image) {
            $serviceData['image_url'] = Storage::url($service->image);
        }

        return response()->json([
            'success' => true,
            'data' => $serviceData
        ], 200);
    }

    /**
     * Get All Available Services
     */
    public function index(Request $request)
    {
        $services = Service::orderBy('name', 'asc')->get();

        // Format services with image URLs
        $servicesData = $services->map(function ($service) {
            $serviceData = $service->toArray();
            if ($service->image) {
                $serviceData['image_url'] = Storage::url($service->image);
            }
            return $serviceData;
        });

        return response()->json([
            'success' => true,
            'data' => $servicesData
        ], 200);
    }

    /**
     * Get Service by Path (for Angular routing)
     */
    public function getByPath($path)
    {
        $service = Service::where('path', '/' . $path)->first();

        if (!$service) {
            return response()->json([
                'success' => false,
                'message' => 'Service not found',
                'is_coming_soon' => false,
                'available' => false
            ], 404);
        }

        $serviceData = [
            'id' => $service->id,
            'name' => $service->name,
            'path' => $service->path,
            'is_coming_soon' => $service->is_coming_soon,
            'available' => !$service->is_coming_soon,
            'message' => $service->is_coming_soon ? 'This service will come soon' : 'Service is available'
        ];

        if ($service->image) {
            $serviceData['image_url'] = Storage::url($service->image);
        }

        return response()->json([
            'success' => true,
            'data' => $serviceData
        ], 200);
    }
}
