<?php

namespace App\Services;

use App\Models\User;
use App\Models\ReferralSetting;
use App\Models\ReferralRelationship;
use App\Models\ReferralEarning;
use Illuminate\Support\Facades\DB;

class ReferralService
{
    /**
     * Process referral when a new user signs up
     */
    public static function processSignupReferral(User $newUser, $referralCode = null)
    {
        $settings = ReferralSetting::getActive();
        if (!$settings || !$settings->is_active) {
            return;
        }

        // Generate referral code for new user
        if (!$newUser->referral_code) {
            $newUser->referral_code = User::generateReferralCode();
            $newUser->save();
        }

        // If no referral code provided, skip
        if (!$referralCode) {
            return;
        }

        // Find referrer by code
        $referrer = User::where('referral_code', $referralCode)->first();
        if (!$referrer) {
            return;
        }

        // Set direct referrer
        $newUser->referred_by = $referrer->id;
        $newUser->save();

        // Create referral relationships for all levels
        self::createReferralRelationships($referrer, $newUser, 1, $settings->referral_levels);

        // Process signup bonus
        if ($settings->signup_bonus > 0) {
            self::createEarning(
                $referrer,
                $newUser,
                1,
                'signup',
                null,
                null,
                0,
                $settings->signup_bonus
            );
        }
    }

    /**
     * Create referral relationships for all levels
     */
    private static function createReferralRelationships($referrer, $referredUser, $currentLevel, $maxLevels)
    {
        if ($currentLevel > $maxLevels || !$referrer) {
            return;
        }

        // Create relationship
        ReferralRelationship::firstOrCreate([
            'referrer_id' => $referrer->id,
            'referred_id' => $referredUser->id,
            'level' => $currentLevel,
        ]);

        // Process next level if referrer has a referrer
        if ($referrer->referred_by && $currentLevel < $maxLevels) {
            $nextLevelReferrer = User::find($referrer->referred_by);
            if ($nextLevelReferrer) {
                self::createReferralRelationships($nextLevelReferrer, $referredUser, $currentLevel + 1, $maxLevels);
            }
        }
    }

    /**
     * Process referral earnings for a transaction (e.g., top-up)
     */
    public static function processTransactionReferral($referredUser, $amount, $sourceId, $sourceType = 'TopUpRequest')
    {
        $settings = ReferralSetting::getActive();
        if (!$settings || !$settings->is_active) {
            return;
        }

        // Get all referrers in the chain
        $referrers = self::getReferrerChain($referredUser, $settings->referral_levels);

        foreach ($referrers as $level => $referrer) {
            $commissionRate = $settings->getCommissionForLevel($level);
            
            if ($commissionRate > 0) {
                $commissionAmount = self::calculateCommission($amount, $commissionRate, $settings->commission_type);
                
                if ($commissionAmount > 0) {
                    self::createEarning(
                        $referrer,
                        $referredUser,
                        $level,
                        'topup',
                        $sourceId,
                        $sourceType,
                        $amount,
                        $commissionAmount
                    );
                }
            }
        }
    }

    /**
     * Get referrer chain up to max levels
     */
    private static function getReferrerChain(User $user, $maxLevels)
    {
        $referrers = [];
        $currentUser = $user;
        $level = 1;

        while ($currentUser->referred_by && $level <= $maxLevels) {
            $referrer = User::find($currentUser->referred_by);
            if ($referrer) {
                $referrers[$level] = $referrer;
                $currentUser = $referrer;
                $level++;
            } else {
                break;
            }
        }

        return $referrers;
    }

    /**
     * Calculate commission amount
     */
    private static function calculateCommission($amount, $rate, $type)
    {
        if ($type === 'percentage') {
            return ($amount * $rate) / 100;
        } else {
            return $rate;
        }
    }

    /**
     * Create earning record
     */
    private static function createEarning($referrer, $referredUser, $level, $type, $sourceId, $sourceType, $amount, $commissionAmount)
    {
        ReferralEarning::create([
            'referrer_id' => $referrer->id,
            'referred_id' => $referredUser->id,
            'level' => $level,
            'earning_type' => $type,
            'source_id' => $sourceId,
            'source_type' => $sourceType,
            'amount' => $amount,
            'commission_amount' => $commissionAmount,
            'status' => 'pending',
        ]);
    }

    /**
     * Credit pending earnings to wallet
     */
    public static function creditEarnings(User $user, $earningIds = null)
    {
        $query = ReferralEarning::where('referrer_id', $user->id)
            ->where('status', 'pending');

        if ($earningIds) {
            $query->whereIn('id', $earningIds);
        }

        $earnings = $query->get();
        $totalAmount = $earnings->sum('commission_amount');

        if ($totalAmount > 0) {
            DB::transaction(function () use ($user, $earnings, $totalAmount) {
                // Update wallet
                $user->wallet_balance += $totalAmount;
                $user->save();

                // Update earnings status
                $earnings->each(function ($earning) {
                    $earning->update([
                        'status' => 'credited',
                        'credited_at' => now(),
                    ]);
                });
            });

            return $totalAmount;
        }

        return 0;
    }
}


