# Offer API Documentation

This document describes the Offer Management APIs for both admin and public endpoints. Offers are general promotional offers, discounts, and deals that can be displayed on the landing page.

---

## Table of Contents

1. [Admin Offer Management APIs](#admin-offer-management-apis)
   - [Get All Offers](#get-all-offers)
   - [Get Single Offer](#get-single-offer)
   - [Add New Offer](#add-new-offer)
   - [Update Offer](#update-offer)
   - [Delete Offer](#delete-offer)
2. [Public Offer APIs](#public-offer-apis)
   - [Get All Active Offers](#get-all-active-offers)
   - [Get Single Active Offer](#get-single-active-offer)
   - [Get Offers by Type](#get-offers-by-type)
   - [Verify Promo Code](#verify-promo-code)

---

## Admin Offer Management APIs

All admin endpoints require authentication using a Bearer token in the Authorization header.

**Base URL:** `/api/admin/offer`

**Authentication:** Required (Admin Bearer Token)

---

### Get All Offers

Get a list of all offers (both active and inactive) with admin details.

**Endpoint:** `GET /api/admin/offer`

**Headers:**
```
Authorization: Bearer {admin_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "data": [
        {
            "id": 1,
            "title": "50% Off on All Services",
            "description": "Get 50% discount on all digital services this month",
            "image_url": "https://example.com/offer1.jpg",
            "offer_type": "service_discount",
            "discount_percentage": 50.00,
            "discount_amount": null,
            "cashback_amount": null,
            "promo_code": "SAVE50",
            "valid_from": "2025-12-01",
            "valid_to": "2025-12-31",
            "link": "https://example.com/services",
            "order": 1,
            "is_active": true,
            "created_by": 1,
            "created_at": "2025-12-06T14:24:00.000000Z",
            "updated_at": "2025-12-06T14:24:00.000000Z",
            "admin": {
                "id": 1,
                "username": "admin",
                "email": "admin@alldigtalseva.com"
            }
        }
    ]
}
```

**Error Response (401) - Unauthorized:**
```json
{
    "success": false,
    "message": "Unauthorized. Admin access required."
}
```

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/admin/offer" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Accept: application/json"
```

---

### Get Single Offer

Get details of a specific offer by ID.

**Endpoint:** `GET /api/admin/offer/{id}`

**URL Parameters:**
- `id` (required, integer): Offer ID

**Headers:**
```
Authorization: Bearer {admin_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "data": {
        "id": 1,
        "title": "50% Off on All Services",
        "description": "Get 50% discount on all digital services this month",
        "image_url": "https://example.com/offer1.jpg",
        "offer_type": "service_discount",
        "discount_percentage": 50.00,
        "discount_amount": null,
        "cashback_amount": null,
        "promo_code": "SAVE50",
        "valid_from": "2025-12-01",
        "valid_to": "2025-12-31",
        "link": "https://example.com/services",
        "order": 1,
        "is_active": true,
        "created_by": 1,
        "created_at": "2025-12-06T14:24:00.000000Z",
        "updated_at": "2025-12-06T14:24:00.000000Z",
        "admin": {
            "id": 1,
            "username": "admin",
            "email": "admin@alldigtalseva.com"
        }
    }
}
```

**Error Response (404) - Not Found:**
```json
{
    "success": false,
    "message": "Offer not found"
}
```

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/admin/offer/1" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Accept: application/json"
```

---

### Add New Offer

Create a new offer.

**Endpoint:** `POST /api/admin/offer`

**Headers:**
```
Authorization: Bearer {admin_token}
Content-Type: application/json
```

**Request Body:**
```json
{
    "title": "50% Off on All Services",
    "description": "Get 50% discount on all digital services this month",
    "image_url": "https://example.com/offer1.jpg",
    "offer_type": "service_discount",
    "discount_percentage": 50.00,
    "discount_amount": null,
    "cashback_amount": null,
    "promo_code": "SAVE50",
    "valid_from": "2025-12-01",
    "valid_to": "2025-12-31",
    "link": "https://example.com/services",
    "order": 1,
    "is_active": true
}
```

**Field Descriptions:**
- `title` (required, string, max 255): Offer title
- `description` (optional, string): Offer description
- `image_url` (optional, string, URL): URL of the offer image
- `offer_type` (optional, enum): Type of offer - `service_discount`, `signup_bonus`, `referral_bonus`, `cashback`, `general` (default: `general`)
- `discount_percentage` (optional, numeric, 0-100): Percentage discount (e.g., 50.00 for 50%)
- `discount_amount` (optional, numeric, min 0): Fixed discount amount in ₹
- `cashback_amount` (optional, numeric, min 0): Cashback amount in ₹
- `promo_code` (optional, string, max 50, unique): Promo code for the offer
- `valid_from` (required, date): Offer start date (YYYY-MM-DD)
- `valid_to` (required, date): Offer end date (YYYY-MM-DD, must be >= valid_from)
- `link` (optional, string, URL): Optional link when offer is clicked
- `order` (optional, integer, min 0): Display order (default: 0)
- `is_active` (optional, boolean): Whether offer is active (default: true)

**Success Response (201):**
```json
{
    "success": true,
    "message": "Offer added successfully",
    "data": {
        "id": 1,
        "title": "50% Off on All Services",
        "description": "Get 50% discount on all digital services this month",
        "image_url": "https://example.com/offer1.jpg",
        "offer_type": "service_discount",
        "discount_percentage": 50.00,
        "discount_amount": null,
        "cashback_amount": null,
        "promo_code": "SAVE50",
        "valid_from": "2025-12-01",
        "valid_to": "2025-12-31",
        "link": "https://example.com/services",
        "order": 1,
        "is_active": true,
        "created_by": 1,
        "created_at": "2025-12-06T14:24:00.000000Z",
        "updated_at": "2025-12-06T14:24:00.000000Z",
        "admin": {
            "id": 1,
            "username": "admin",
            "email": "admin@alldigtalseva.com"
        }
    }
}
```

**Error Response (422) - Validation Error:**
```json
{
    "success": false,
    "message": "Validation error",
    "errors": {
        "title": [
            "The title field is required."
        ],
        "valid_to": [
            "The valid to must be a date after or equal to valid from."
        ],
        "promo_code": [
            "The promo code has already been taken."
        ]
    }
}
```

**cURL Example:**
```bash
curl -X POST "https://api.alldigtalseva.com/api/admin/offer" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "title": "50% Off on All Services",
    "description": "Get 50% discount on all digital services this month",
    "image_url": "https://example.com/offer1.jpg",
    "offer_type": "service_discount",
    "discount_percentage": 50.00,
    "promo_code": "SAVE50",
    "valid_from": "2025-12-01",
    "valid_to": "2025-12-31",
    "link": "https://example.com/services",
    "order": 1,
    "is_active": true
  }'
```

---

### Update Offer

Update an existing offer.

**Endpoint:** `PUT /api/admin/offer/{id}` or `PATCH /api/admin/offer/{id}`

**URL Parameters:**
- `id` (required, integer): Offer ID

**Headers:**
```
Authorization: Bearer {admin_token}
Content-Type: application/json
```

**Request Body:**
```json
{
    "title": "Updated Offer Title",
    "discount_percentage": 60.00,
    "is_active": false
}
```

**Field Descriptions:**
- All fields are optional (use `PUT` for full update, `PATCH` for partial update)
- Same validation rules as Add New Offer

**Success Response (200):**
```json
{
    "success": true,
    "message": "Offer updated successfully",
    "data": {
        "id": 1,
        "title": "Updated Offer Title",
        "discount_percentage": 60.00,
        "is_active": false,
        ...
    }
}
```

**Error Response (404) - Not Found:**
```json
{
    "success": false,
    "message": "Offer not found"
}
```

**cURL Example:**
```bash
curl -X PUT "https://api.alldigtalseva.com/api/admin/offer/1" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "title": "Updated Offer Title",
    "discount_percentage": 60.00,
    "is_active": false
  }'
```

---

### Delete Offer

Delete an offer.

**Endpoint:** `DELETE /api/admin/offer/{id}`

**URL Parameters:**
- `id` (required, integer): Offer ID

**Headers:**
```
Authorization: Bearer {admin_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "message": "Offer deleted successfully"
}
```

**Error Response (404) - Not Found:**
```json
{
    "success": false,
    "message": "Offer not found"
}
```

**cURL Example:**
```bash
curl -X DELETE "https://api.alldigtalseva.com/api/admin/offer/1" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Accept: application/json"
```

---

## Public Offer APIs

Public endpoints for fetching active offers for the landing page. No authentication required.

**Base URL:** `/api/offer`

**Authentication:** Not required

---

### Get All Active Offers

Get all active and valid offers ordered by display order. This endpoint is used for the landing page.

**Endpoint:** `GET /api/offer`

**Success Response (200):**
```json
{
    "success": true,
    "data": [
        {
            "id": 1,
            "title": "50% Off on All Services",
            "description": "Get 50% discount on all digital services this month",
            "image_url": "https://example.com/offer1.jpg",
            "offer_type": "service_discount",
            "discount_percentage": 50.00,
            "discount_amount": null,
            "cashback_amount": null,
            "promo_code": "SAVE50",
            "valid_from": "2025-12-01",
            "valid_to": "2025-12-31",
            "link": "https://example.com/services",
            "order": 1
        },
        {
            "id": 2,
            "title": "Signup Bonus ₹100",
            "description": "Get ₹100 cashback on signup",
            "image_url": "https://example.com/offer2.jpg",
            "offer_type": "signup_bonus",
            "discount_percentage": null,
            "discount_amount": null,
            "cashback_amount": 100.00,
            "promo_code": null,
            "valid_from": "2025-12-01",
            "valid_to": "2025-12-31",
            "link": null,
            "order": 2
        }
    ]
}
```

**Note:** Only active offers (`is_active = true`) that are within their validity period are returned, ordered by `order` field (ascending), then by creation date (descending).

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/offer" \
  -H "Accept: application/json"
```

**JavaScript/Fetch Example:**
```javascript
fetch('https://api.alldigtalseva.com/api/offer')
  .then(response => response.json())
  .then(data => {
    if (data.success) {
      console.log('Offers:', data.data);
      // Display offers on landing page
    }
  })
  .catch(error => console.error('Error:', error));
```

---

### Get Single Active Offer

Get details of a specific active offer by ID.

**Endpoint:** `GET /api/offer/{id}`

**URL Parameters:**
- `id` (required, integer): Offer ID

**Success Response (200):**
```json
{
    "success": true,
    "data": {
        "id": 1,
        "title": "50% Off on All Services",
        "description": "Get 50% discount on all digital services this month",
        "image_url": "https://example.com/offer1.jpg",
        "offer_type": "service_discount",
        "discount_percentage": 50.00,
        "discount_amount": null,
        "cashback_amount": null,
        "promo_code": "SAVE50",
        "valid_from": "2025-12-01",
        "valid_to": "2025-12-31",
        "link": "https://example.com/services",
        "order": 1
    }
}
```

**Error Response (404) - Not Found or Expired:**
```json
{
    "success": false,
    "message": "Offer not found or expired"
}
```

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/offer/1" \
  -H "Accept: application/json"
```

---

### Get Offers by Type

Get all active offers filtered by offer type.

**Endpoint:** `GET /api/offer/type/{type}`

**URL Parameters:**
- `type` (required, string): Offer type - `service_discount`, `signup_bonus`, `referral_bonus`, `cashback`, or `general`

**Success Response (200):**
```json
{
    "success": true,
    "data": [
        {
            "id": 1,
            "title": "50% Off on All Services",
            "description": "Get 50% discount on all digital services this month",
            "image_url": "https://example.com/offer1.jpg",
            "offer_type": "service_discount",
            "discount_percentage": 50.00,
            "promo_code": "SAVE50",
            "valid_from": "2025-12-01",
            "valid_to": "2025-12-31",
            "link": "https://example.com/services",
            "order": 1
        }
    ]
}
```

**Error Response (400) - Invalid Type:**
```json
{
    "success": false,
    "message": "Invalid offer type. Valid types: service_discount, signup_bonus, referral_bonus, cashback, general"
}
```

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/offer/type/service_discount" \
  -H "Accept: application/json"
```

---

### Verify Promo Code

Verify if a promo code is valid and get offer details.

**Endpoint:** `POST /api/offer/verify-promo`

**Request Body:**
```json
{
    "promo_code": "SAVE50"
}
```

**Field Descriptions:**
- `promo_code` (required, string): The promo code to verify

**Success Response (200):**
```json
{
    "success": true,
    "message": "Promo code is valid",
    "data": {
        "id": 1,
        "title": "50% Off on All Services",
        "description": "Get 50% discount on all digital services this month",
        "image_url": "https://example.com/offer1.jpg",
        "offer_type": "service_discount",
        "discount_percentage": 50.00,
        "discount_amount": null,
        "cashback_amount": null,
        "promo_code": "SAVE50",
        "valid_from": "2025-12-01",
        "valid_to": "2025-12-31",
        "link": "https://example.com/services"
    }
}
```

**Error Response (404) - Invalid or Expired:**
```json
{
    "success": false,
    "message": "Invalid or expired promo code"
}
```

**Error Response (422) - Validation Error:**
```json
{
    "success": false,
    "message": "Validation error",
    "errors": {
        "promo_code": [
            "The promo code field is required."
        ]
    }
}
```

**cURL Example:**
```bash
curl -X POST "https://api.alldigtalseva.com/api/offer/verify-promo" \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "promo_code": "SAVE50"
  }'
```

**JavaScript/Fetch Example:**
```javascript
async function verifyPromoCode(promoCode) {
  try {
    const response = await fetch('https://api.alldigtalseva.com/api/offer/verify-promo', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ promo_code: promoCode })
    });
    
    const result = await response.json();
    
    if (result.success) {
      console.log('Valid promo code:', result.data);
      return result.data;
    } else {
      console.error('Invalid promo code:', result.message);
      return null;
    }
  } catch (error) {
    console.error('Error verifying promo code:', error);
    return null;
  }
}
```

---

## Offer Types

| Type | Description |
|------|-------------|
| `service_discount` | Discount on services |
| `signup_bonus` | Bonus for new user signups |
| `referral_bonus` | Bonus for referrals |
| `cashback` | Cashback offers |
| `general` | General promotional offers |

---

## Integration Notes

### For Landing Page Integration

1. **Fetch Offers on Page Load:**
   ```javascript
   // Angular/React/Vue example
   async function loadOffers() {
     try {
       const response = await fetch('https://api.alldigtalseva.com/api/offer');
       const result = await response.json();
       
       if (result.success) {
         // result.data contains array of active offers
         displayOffers(result.data);
       }
     } catch (error) {
       console.error('Failed to load offers:', error);
     }
   }
   ```

2. **Display Order:**
   - Offers are automatically ordered by the `order` field (ascending)
   - Lower order numbers appear first
   - If order is the same, newer offers appear first

3. **Image URLs:**
   - Always use the `image_url` field for displaying offer images
   - Ensure images are optimized for web (recommended: WebP format, compressed)

4. **Optional Links:**
   - If `link` is provided, make the offer clickable
   - If `link` is `null`, offer is display-only

5. **Promo Code Verification:**
   - Use the verify promo code endpoint before applying discounts
   - Always verify on the backend when processing payments

### Best Practices

1. **Image Optimization:**
   - Use high-quality images (recommended: 800x600 or similar aspect ratio)
   - Compress images before uploading
   - Use CDN for faster loading

2. **Order Management:**
   - Use order values like 1, 2, 3 for sequential display
   - Leave gaps (e.g., 10, 20, 30) for easy reordering

3. **Active Status:**
   - Set `is_active: false` to temporarily hide offers without deleting
   - Useful for seasonal promotions or A/B testing

4. **Validity Dates:**
   - Always set appropriate validity dates
   - System automatically filters expired offers

5. **Promo Codes:**
   - Use uppercase letters and numbers
   - Keep codes short and memorable
   - Ensure uniqueness

---

## Error Codes

| Status Code | Description |
|------------|-------------|
| 200 | Success |
| 201 | Created successfully |
| 400 | Bad request |
| 401 | Unauthorized (Admin token required) |
| 404 | Resource not found |
| 422 | Validation error |

---

## Support

For issues or questions regarding the Offer API, please contact the development team.

