const puppeteer = require('puppeteer-core');
const fs = require('fs');
const path = require('path');
const axios = require('axios');
const FormData = require('form-data');

// OCR.space API configuration (Free OCR service)
const OCR_SPACE_API_KEY = '08a786f7ae88957';
const OCR_SPACE_URL = 'https://api.ocr.space/parse/image';

class TokenManager {
  constructor() {
    this.token = null;
    this.tokenExpiry = null;
    this.tokenFile = path.join(__dirname, '../token-cache.json');
    this.loadTokenFromCache();
  }

  // Wait for authentication to complete before attempting token extraction
  async waitForAuthentication(page, config, timeout = 45000) {
    console.log('Waiting for authentication to complete...');
    const startTime = Date.now();

    while (Date.now() - startTime < timeout) {
      try {
        const currentUrl = page.url();
        console.log(`Current URL: ${currentUrl}`);

        // Check if we've been redirected to target domain
        if (currentUrl.includes(config.includes_pattern)) {
          console.log(`Successfully redirected to target domain (${config.includes_pattern}) - checking for token...`);

          // Check if the token is available in localStorage
          const tokenCheck = await page.evaluate(() => {
            const token = localStorage.getItem('token');
            if (token) {
              return { type: 'token_found', token: token.substring(0, 20) + '...' };
            }
            return { type: 'waiting', message: 'Token not yet available in localStorage' };
          });

          if (tokenCheck.type === 'token_found') {
            console.log('✅ Token found in localStorage at target domain');
            return true;
          }

          console.log('Token not yet available, waiting...');
          await new Promise(resolve => setTimeout(resolve, 2000));
          continue;
        }

        // Check if we've moved past the authorization page
        if (!currentUrl.includes('authorize')) {
          console.log('Successfully moved past authorization page');
          return true;
        }

        // Check for success indicators in the page
        const authStatus = await page.evaluate(() => {
          // Look for success messages, redirects, or token presence
          const successElements = document.querySelectorAll('.success, .alert-success, [class*="success"], [data-auth="success"]');
          for (const el of successElements) {
            if (el.textContent && el.textContent.trim()) {
              return { type: 'success', message: el.textContent.trim() };
            }
          }

          // Look for error messages
          const errorElements = document.querySelectorAll('.error, .alert-danger, [class*="error"], [data-auth="error"]');
          for (const el of errorElements) {
            if (el.textContent && el.textContent.trim()) {
              return { type: 'error', message: el.textContent.trim() };
            }
          }

          // Check for redirect indicators
          const metaRefresh = document.querySelector('meta[http-equiv="refresh"]');
          if (metaRefresh) {
            return { type: 'redirect_pending', url: metaRefresh.getAttribute('content') };
          }

          return { type: 'waiting', message: 'Still on authorization page' };
        });

        console.log('Authentication status:', authStatus);

        // If we found an error, throw it
        if (authStatus.type === 'error') {
          throw new Error(`Authentication failed: ${authStatus.message}`);
        }

        // If we're waiting for redirect, give it more time
        if (authStatus.type === 'redirect_pending') {
          console.log('Redirect pending, waiting...');
          await new Promise(resolve => setTimeout(resolve, 2000));
          continue;
        }

        // Wait a bit before checking again
        await new Promise(resolve => setTimeout(resolve, 2000));

      } catch (error) {
        console.error('Error during authentication wait:', error.message);
        // Don't throw here, just continue waiting
      }
    }

    console.log('Authentication timeout reached');
    return false;
  }

  // Load cached token if available
  loadTokenFromCache() {
    try {
      if (fs.existsSync(this.tokenFile)) {
        const cached = JSON.parse(fs.readFileSync(this.tokenFile, 'utf8'));
        if (cached.token && cached.expiry && new Date() < new Date(cached.expiry)) {
          this.token = cached.token;
          this.tokenExpiry = new Date(cached.expiry);
          console.log('Token loaded from cache, expires at:', this.tokenExpiry);
        }
      }
    } catch (error) {
      console.error('Error loading cached token:', error);
    }
  }

  // Save token to cache
  saveTokenToCache() {
    try {
      const cacheData = {
        token: this.token,
        expiry: this.tokenExpiry,
        cachedAt: new Date().toISOString()
      };
      fs.writeFileSync(this.tokenFile, JSON.stringify(cacheData, null, 2));
    } catch (error) {
      console.error('Error saving token to cache:', error);
    }
  }

  // Check if token is expired or will expire soon (within 1 hour)
  isTokenExpired() {
    if (!this.token || !this.tokenExpiry) return true;

    const now = new Date();
    const oneHourFromNow = new Date(now.getTime() + 60 * 60 * 1000);

    return this.tokenExpiry <= oneHourFromNow;
  }

  // Get valid token (refresh if needed)
  async getValidToken(config) {
    if (this.isTokenExpired()) {
      console.log('Token expired or expiring soon, refreshing...');
      try {
        await this.refreshToken(config);
      } catch (error) {
        console.error('Failed to refresh token:', error.message);

        // If we have a cached token that's not too old, try to use it
        if (this.token && this.tokenExpiry) {
          const now = new Date();
          const tokenAge = now - this.tokenExpiry;
          const maxAge = 12 * 60 * 60 * 1000; // 12 hours

          if (tokenAge < maxAge) {
            console.log('Using cached token despite refresh failure');
            return this.token;
          }
        }

        // If all else fails, throw the error
        throw new Error(`Token refresh failed: ${error.message}`);
      }
    }
    return this.token;
  }

  // Extract token from the website (this will be implemented based on your login process)
  async refreshToken(config = {}) {
    try {
      console.log('Starting token refresh process...');

      // Configuration from argument or defaults
      const authConfig = {
        username: config.username || process.env.CSC_USERNAME || '516821460010',
        password: config.password || process.env.CSC_PASSWORD || 'Vaishnavi@98',
        url: config.url || 'https://mhfr.agristack.gov.in/farmer-registry-mh/#/',
        includes_pattern: config.includes_pattern || 'mhfr.agristack.gov.in'
      };

      if (authConfig.username === 'your_username_here' || authConfig.password === 'your_password_here') {
        throw new Error('Please set credentials in config or environment variables');
      }

      console.log('Launching browser...');

      // Try to find Chrome/Chromium executable
      let executablePath = null;

      // Common Chrome paths on different systems
      const chromePaths = [
        '/usr/bin/google-chrome', // Linux
        '/usr/bin/chromium-browser', // Linux
        'C:\\Program Files\\Google\\Chrome\\Application\\chrome.exe', // Windows
        'C:\\Program Files (x86)\\Google\\Chrome\\Application\\chrome.exe', // Windows
        '/Applications/Google Chrome.app/Contents/MacOS/Google Chrome' // macOS
      ];

      for (const path of chromePaths) {
        if (fs.existsSync(path)) {
          executablePath = path;
          break;
        }
      }

      if (!executablePath) {
        throw new Error('Chrome/Chromium not found. Please install Chrome or specify the executable path.');
      }

      console.log('Using Chrome at:', executablePath);

      const browser = await puppeteer.launch({
        headless: true, // Set to false for debugging and manual captcha solving
        executablePath: executablePath,
        args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-dev-shm-usage'],
        slowMo: 50 // Reduced for faster operation
      });

      try {
        const page = await browser.newPage();

        // Step 1: Navigate to main site
        console.log(`Navigating to ${authConfig.url}...`);
        await page.goto(authConfig.url, { waitUntil: 'networkidle0' });

        // Debug: Log page title and content
        const pageTitle = await page.title();
        console.log('Page title:', pageTitle);

        // Debug: Log all links and buttons on the page
        const pageElements = await page.evaluate(() => {
          const links = Array.from(document.querySelectorAll('a')).map(a => ({
            text: a.textContent?.trim(),
            href: a.href,
            onclick: a.onclick?.toString()
          }));
          const buttons = Array.from(document.querySelectorAll('button')).map(b => ({
            text: b.textContent?.trim(),
            onclick: b.onclick?.toString()
          }));
          return { links, buttons };
        });
        console.log('Page elements:', JSON.stringify(pageElements, null, 2));

        // Step 2: Click "Login with CSC" button
        console.log('Clicking Login with CSC...');

        // Wait for page to load completely (reduced wait time)
        await new Promise(resolve => setTimeout(resolve, 1500));

        // Try multiple selectors for the Login with CSC button
        let loginButton = null;
        const selectors = [
          'a[href*="csc"]',
          '[onclick*="csc"]',
          'a[onclick*="csc"]'
        ];

        for (const selector of selectors) {
          try {
            loginButton = await page.$(selector);
            if (loginButton) {
              console.log('Found login button with selector:', selector);
              break;
            }
          } catch (e) {
            // Continue to next selector
          }
        }

        if (!loginButton) {
          // Try to find by text content
          loginButton = await page.evaluateHandle(() => {
            const elements = Array.from(document.querySelectorAll('a, button'));
            return elements.find(el =>
              el.textContent && el.textContent.toLowerCase().includes('login with csc')
            );
          });
        }

        if (!loginButton) {
          throw new Error('Could not find Login with CSC button');
        }

        // Click the button using page.evaluate
        await page.evaluate((button) => button.click(), loginButton);
        console.log('Clicked Login with CSC button');

        // Step 3: Wait for popup and click "Okay"
        console.log('Waiting for popup and clicking Okay...');
        await new Promise(resolve => setTimeout(resolve, 1000)); // Wait for popup to appear (reduced)

        // Try to find and click the Okay button in popup
        const popupButtons = await page.$$('button');
        let okayClicked = false;
        for (const button of popupButtons) {
          const text = await button.evaluate(el => el.textContent);
          if (text && text.toLowerCase().includes('okay')) {
            await button.click();
            okayClicked = true;
            break;
          }
        }

        if (!okayClicked) {
          console.log('Okay button not found, trying alternative approach...');
          // Try clicking any button that might be the popup button
          await new Promise(resolve => setTimeout(resolve, 1000));
        }

        // Step 4: Wait for redirect to Digital Seva Connect
        console.log('Waiting for redirect to Digital Seva Connect...');
        await new Promise(resolve => setTimeout(resolve, 5000));

        // Check current URL to see where we are
        const cscRedirectUrl = page.url();
        console.log('Current URL after CSC click:', cscRedirectUrl);

        // Step 5: Login to Digital Seva Connect
        console.log('Logging into Digital Seva Connect...');

        // Wait for either the login form or a redirect
        try {
          await page.waitForSelector('input[placeholder*="Username"], input[placeholder*="Email"]', { timeout: 10000 });
          console.log('Login form found');
        } catch (e) {
          console.log('Login form not found, checking if we need to wait longer...');
          await new Promise(resolve => setTimeout(resolve, 5000));

          // Check URL again
          const newUrl = page.url();
          console.log('URL after additional wait:', newUrl);

          if (!newUrl.includes('connect.csc.gov.in')) {
            throw new Error('Failed to reach Digital Seva Connect login page');
          }
        }

        // Fill username (faster typing)
        await page.type('input[placeholder*="Username"], input[placeholder*="Email"]', authConfig.username, { delay: 10 });

        // Fill password (faster typing)
        await page.type('input[type="password"]', authConfig.password, { delay: 10 });

        // Handle captcha with OCR
        console.log('Handling captcha...');
        const captchaInput = await page.$('input[placeholder*="Captcha"]');
        if (captchaInput) {
          console.log('Captcha detected - attempting OCR solution...');

          let captchaImage = null;
          let captchaScreenshot = null;

          try {
            // Find the captcha image - look for the specific structure from the HTML
            // The captcha is an img tag with data:image/jpeg;base64 source
            captchaImage = await page.$('img[src*="data:image/jpeg;base64"]');

            if (!captchaImage) {
              // Fallback: try other selectors
              const captchaSelectors = [
                'img[src*="captcha"]',
                'img[alt*="captcha"]',
                'img[class*="captcha"]',
                'img[src*="image"]'
              ];

              for (const selector of captchaSelectors) {
                captchaImage = await page.$(selector);
                if (captchaImage) {
                  console.log('Found captcha image with selector:', selector);
                  break;
                }
              }
            } else {
              console.log('Found captcha image with base64 data source');
            }

            if (captchaImage) {
              // Take screenshot of the captcha image
              captchaScreenshot = await captchaImage.screenshot();

              // Use human captcha solving system
              console.log('Using human captcha solving system...');

              try {
                // Convert captcha image to base64 for upload
                const captchaBase64 = captchaScreenshot.toString('base64');

                // Upload captcha to backend for human solving
                console.log('Uploading captcha to backend for human solving...');
                const sessionId = `captcha_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

                const uploadResponse = await axios.post('http://localhost:4000/api/farmer/captcha/upload', {
                  captchaImage: `data:image/jpeg;base64,${captchaBase64}`,
                  sessionId: sessionId
                });

                console.log('Captcha uploaded with session ID:', sessionId);
                console.log('Please solve the captcha in the Angular website popup');

                // Wait for human to solve the captcha
                console.log('Waiting for human to solve captcha...');

                let captchaSolved = false;
                let solvedValue = null;
                const maxWaitTime = 5 * 60 * 1000; // 5 minutes
                const startTime = Date.now();

                while (!captchaSolved && (Date.now() - startTime) < maxWaitTime) {
                  try {
                    const statusResponse = await axios.get(`http://localhost:4000/api/farmer/captcha/status/${sessionId}`);

                    if (statusResponse.data.status === 'solved') {
                      captchaSolved = true;
                      solvedValue = statusResponse.data.solvedValue;
                      console.log('Captcha solved by human:', solvedValue);
                      break;
                    }

                    // Wait 1 second before checking again (faster checking)
                    await new Promise(resolve => setTimeout(resolve, 1000));
                  } catch (statusError) {
                    console.log('Checking captcha status...');
                    await new Promise(resolve => setTimeout(resolve, 1000));
                  }
                }

                if (captchaSolved && solvedValue) {
                  // Type the solved captcha value
                  await captchaInput.type(solvedValue);
                  console.log('Human-solved captcha entered successfully');
                } else {
                  throw new Error('Captcha solving timeout or failed');
                }

              } catch (uploadError) {
                console.error('Failed to use human captcha solving:', uploadError.message);
                throw uploadError; // This will trigger the manual fallback
              }
            } else {
              throw new Error('Could not find captcha image');
            }

          } catch (ocrError) {
            console.error('OCR failed:', ocrError.message);
            console.log('Falling back to human captcha solving...');

            if (captchaScreenshot) {
              try {
                // Convert captcha image to base64 for upload
                const captchaBase64 = captchaScreenshot.toString('base64');

                // Upload captcha to our website for human solving
                console.log('Uploading captcha for human solving...');
                const uploadResponse = await axios.post('http://localhost:3000/api/farmer/captcha/upload', {
                  captchaImage: `data:image/jpeg;base64,${captchaBase64}`
                });
                const { sessionId } = uploadResponse.data;
                let captchaSolved = false;
                let solvedValue = null;
                const maxWaitTime = 100 * 60 * 1000; // 5 minutes
                const startTime = Date.now();
                while (!captchaSolved && (Date.now() - startTime) < maxWaitTime) {
                  try {
                    const statusResponse = await axios.get(`http://localhost:3000/api/farmer/captcha/status/${sessionId}`);
                    if (statusResponse.data.status === 'solved') {
                      captchaSolved = true;
                      solvedValue = statusResponse.data.solvedValue;
                      console.log('Captcha solved by human:', solvedValue);
                      break;
                    }

                    // Wait 1 second before checking again (faster checking)
                    await new Promise(resolve => setTimeout(resolve, 1000));

                  } catch (statusError) {
                    console.log('Checking captcha status...');
                    await new Promise(resolve => setTimeout(resolve, 1000));
                  }
                }

                if (captchaSolved && solvedValue) {
                  // Type the solved captcha value (faster typing)
                  await captchaInput.type(solvedValue, { delay: 10 });
                  console.log('Human-solved captcha entered successfully');
                } else {
                  throw new Error('Captcha solving timeout or failed');
                }

              } catch (uploadError) {
                console.error('Failed to use human captcha solving:', uploadError.message);
                console.log('Falling back to manual browser input...');

                // Fallback: wait for manual input in browser
                console.log('Please solve the captcha manually in the browser...');
                await new Promise(resolve => setTimeout(resolve, 30000)); // 30 seconds for manual input
              }
            } else {
              console.log('No captcha screenshot available, waiting for manual input...');
              await new Promise(resolve => setTimeout(resolve, 30000)); // 30 seconds for manual input
            }
          }
        }

        // Click Sign In - using the correct HTML structure
        console.log('Looking for Sign In button...');

        // Based on the actual HTML: <input type="submit" name="login" value="Sign In" class="loginBtn">
        let signInClicked = false;

        // Strategy 1: Look for input[type="submit"] with name="login"
        try {
          await page.click('input[type="submit"][name="login"]');
          signInClicked = true;
          console.log('Clicked Sign In button by input[type="submit"][name="login"]');
        } catch (e) {
          console.log('Strategy 1 failed, trying alternatives...');
        }

        // Strategy 2: Look for input with class="loginBtn"
        if (!signInClicked) {
          try {
            await page.click('input.loginBtn');
            signInClicked = true;
            console.log('Clicked Sign In button by class="loginBtn"');
          } catch (e) {
            console.log('Strategy 2 failed, trying alternatives...');
          }
        }

        // Strategy 3: Look for input with value="Sign In"
        if (!signInClicked) {
          try {
            await page.click('input[value="Sign In"]');
            signInClicked = true;
            console.log('Clicked Sign In button by value="Sign In"');
          } catch (e) {
            console.log('Strategy 3 failed, trying alternatives...');
          }
        }

        // Strategy 4: Fallback - look for any submit input
        if (!signInClicked) {
          try {
            await page.click('input[type="submit"]');
            signInClicked = true;
            console.log('Clicked Sign In button by input[type="submit"]');
          } catch (e) {
            console.log('Strategy 4 failed, trying alternatives...');
          }
        }

        // Strategy 5: Last resort - look for any element containing "Sign In"
        if (!signInClicked) {
          try {
            const signInElement = await page.evaluate(() => {
              const elements = Array.from(document.querySelectorAll('input, button'));
              return elements.find(el =>
                el.value && el.value.toLowerCase().includes('sign in') ||
                el.textContent && el.textContent.toLowerCase().includes('sign in')
              );
            });

            if (signInElement) {
              await page.evaluate((el) => el.click(), signInElement);
              signInClicked = true;
              console.log('Clicked Sign In button by text content search');
            }
          } catch (e) {
            console.log('Strategy 5 failed');
          }
        }

        if (!signInClicked) {
          throw new Error('Could not find or click Sign In button');
        }

        // Step 6: Wait for authentication to complete with proper validation
        console.log('Waiting for authentication to complete...');

        // Use the new authentication validation method
        const authCompleted = await this.waitForAuthentication(page, authConfig, 45000);

        if (!authCompleted) {
          // Get detailed page information for debugging
          const pageContent = await page.evaluate(() => {
            return {
              title: document.title,
              url: window.location.href,
              bodyText: document.body.textContent.substring(0, 500),
              localStorage: Object.keys(localStorage),
              sessionStorage: Object.keys(sessionStorage),
              cookies: document.cookie
            };
          });

          console.log('Authentication timeout - page details:', pageContent);

          // Instead of crashing, try to provide more helpful error information
          const errorDetails = {
            message: 'Authentication did not complete within the expected time',
            possibleCauses: [
              'CSC system is experiencing delays',
              'Authentication flow has changed',
              'Network connectivity issues',
              'Captcha validation problems'
            ],
            pageState: pageContent,
            timestamp: new Date().toISOString()
          };

          console.error('Authentication failed with details:', JSON.stringify(errorDetails, null, 2));

          // Throw a more descriptive error that won't crash the entire application
          throw new Error(`Authentication failed: ${errorDetails.message}. Check logs for details.`);
        }

        console.log('Authentication completed successfully, proceeding to token extraction...');

        // First, check if we're on AgriStack domain and can extract token directly
        console.log('Checking if we can extract token directly from AgriStack domain...');
        const agriStackCheck = await this.checkAgriStackToken(page);

        let token = null;

        if (agriStackCheck.found) {
          console.log('✅ Token extracted directly from AgriStack domain');
          token = agriStackCheck.token;
        } else {
          console.log('Extracting token using fallback methods...');
          // Extract token from network requests or local storage using fallback methods
          token = await this.extractToken(page);
        }

        if (token) {
          this.token = token;
          this.tokenExpiry = new Date(Date.now() + 11 * 60 * 60 * 1000); // 11 hours (refresh before 12 hour expiry)
          this.saveTokenToCache();
          console.log('Token successfully refreshed!');
        } else {
          // Try to get more information about what happened
          const pageContent = await page.evaluate(() => {
            return {
              title: document.title,
              url: window.location.href,
              bodyText: document.body.textContent.substring(0, 500)
            };
          });

          console.log('Page content after login:', pageContent);
          throw new Error('Could not extract token from the authenticated session. Please check the logs for page content.');
        }

      } finally {
        await browser.close();
      }

    } catch (error) {
      console.error('Error refreshing token:', error);
      throw error;
    }
  }

  // OCR.Space OCR function for captcha recognition
  async recognizeCaptchaWithOCRSpace(imageBuffer) {
    try {
      console.log('Using OCR.Space for captcha recognition...');

      // Convert buffer to base64
      const base64Image = imageBuffer.toString('base64');

      // Prepare form data for OCR.Space API
      const formData = new URLSearchParams();
      formData.append('apikey', OCR_SPACE_API_KEY);
      formData.append('base64Image', `data:image/jpeg;base64,${base64Image}`);
      formData.append('language', 'eng');
      formData.append('isOverlayRequired', 'false');
      formData.append('filetype', 'jpg');
      formData.append('detectOrientation', 'false');
      formData.append('scale', 'true');
      formData.append('OCREngine', '2'); // Engine 2 is best for numbers

      // Make request to OCR.Space API
      const response = await axios.post(OCR_SPACE_URL, formData, {
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded'
        },
        timeout: 30000 // 30 second timeout
      });

      console.log('OCR.Space API Response Status:', response.status);
      console.log('OCR.Space API Response Data:', JSON.stringify(response.data, null, 2));

      if (response.data && response.data.ParsedResults && response.data.ParsedResults.length > 0) {
        const recognizedText = response.data.ParsedResults[0].ParsedText;
        console.log('OCR.Space raw result:', recognizedText);

        // Extract only numbers from the recognized text
        const numbersOnly = recognizedText.replace(/[^0-9]/g, '');
        console.log('OCR.Space numbers extracted:', numbersOnly);

        return numbersOnly;
      } else if (response.data && response.data.ErrorMessage) {
        throw new Error(`OCR.Space API Error: ${response.data.ErrorMessage}`);
      } else {
        throw new Error('OCR.Space API response format unexpected');
      }

    } catch (error) {
      if (error.response) {
        console.error('OCR.Space API Error Response:', error.response.status, error.response.data);
      }
      console.error('OCR.Space OCR failed:', error.message);
      throw error;
    }
  }

  // OCR.space OCR function for captcha recognition
  async recognizeCaptchaWithOCRspace(imageBuffer) {
    try {
      console.log('Using OCR.space for captcha recognition...');

      // Create form data for OCR.space API
      const form = new FormData();
      form.append('apikey', OCR_SPACE_API_KEY);
      form.append('image', imageBuffer, {
        filename: 'captcha.jpg',
        contentType: 'image/jpeg'
      });
      form.append('language', 'eng');
      form.append('isOverlayRequired', 'false');
      form.append('filetype', 'jpg');
      form.append('detectOrientation', 'false');
      form.append('scale', 'true');
      form.append('OCREngine', '1'); // Use engine 1 for better compatibility

      // Make request to OCR.space API
      const response = await axios.post(OCR_SPACE_URL, form, {
        headers: {
          ...form.getHeaders()
        },
        timeout: 30000 // 30 second timeout
      });

      console.log('OCR.space API Response Status:', response.status);

      if (response.data && response.data.ParsedResults && response.data.ParsedResults.length > 0) {
        const recognizedText = response.data.ParsedResults[0].ParsedText;
        console.log('OCR.space raw result:', recognizedText);

        // Extract only numbers from the recognized text
        const numbersOnly = recognizedText.replace(/[^0-9]/g, '');
        console.log('OCR.space numbers extracted:', numbersOnly);

        return numbersOnly;
      } else if (response.data && response.data.ErrorMessage) {
        throw new Error(`OCR.space API Error: ${response.data.ErrorMessage}`);
      } else {
        throw new Error('OCR.space API response format unexpected');
      }

    } catch (error) {
      if (error.response) {
        console.error('OCR.space API Error Response:', error.response.status, error.response.data);
      }
      console.error('OCR.space OCR failed:', error.message);
      throw error;
    }
  }

  // Extract token from the authenticated page
  async extractToken(page) {
    try {
      // Method 1: Check for the specific "token" key in localStorage (priority)
      console.log('Checking for token in localStorage with key "token"...');
      const localStorageToken = await page.evaluate(() => {
        return localStorage.getItem('token');
      });

      if (localStorageToken) {
        console.log('✅ Token found in localStorage with key "token"');
        return localStorageToken;
      }

      // Method 2: Check other localStorage keys as fallback
      const fallbackToken = await page.evaluate(() => {
        return localStorage.getItem('authToken') ||
          localStorage.getItem('bearerToken') ||
          localStorage.getItem('accessToken');
      });

      if (fallbackToken) {
        console.log('Token found in localStorage with fallback key');
        return fallbackToken;
      }

      // Method 3: Check session storage
      const sessionStorageToken = await page.evaluate(() => {
        return sessionStorage.getItem('token') ||
          sessionStorage.getItem('authToken') ||
          sessionStorage.getItem('bearerToken') ||
          sessionStorage.getItem('accessToken');
      });

      if (sessionStorageToken) {
        console.log('Token found in sessionStorage');
        return sessionStorageToken;
      }

      // Method 4: Check cookies
      const cookies = await page.cookies();
      const tokenCookie = cookies.find(cookie =>
        cookie.name.toLowerCase().includes('token') ||
        cookie.name.toLowerCase().includes('auth')
      );

      if (tokenCookie) {
        console.log('Token found in cookies');
        return tokenCookie.value;
      }

      // Method 5: Monitor network requests for authorization headers
      console.log('Monitoring network requests for token...');
      const token = await page.evaluate(() => {
        return new Promise((resolve) => {
          const originalFetch = window.fetch;
          let foundToken = null;

          window.fetch = function (...args) {
            const request = args[0];
            const options = args[1] || {};

            if (options.headers && options.headers.authorization) {
              foundToken = options.headers.authorization.replace('Bearer ', '');
            }

            return originalFetch.apply(this, args);
          };

          // Wait a bit for any requests to complete
          setTimeout(() => resolve(foundToken), 3000);
        });
      });

      if (token) {
        console.log('Token found in network requests');
        return token;
      }

      // Method 6: Check URL parameters for OAuth tokens
      console.log('Checking URL parameters for OAuth tokens...');
      const urlToken = await page.evaluate(() => {
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.get('access_token') || urlParams.get('token') || urlParams.get('code');
      });

      if (urlToken) {
        console.log('Token found in URL parameters');
        return urlToken;
      }

      // Method 7: Check for any hidden input fields containing tokens
      console.log('Checking hidden input fields for tokens...');
      const hiddenToken = await page.evaluate(() => {
        const hiddenInputs = Array.from(document.querySelectorAll('input[type="hidden"]'));
        for (const input of hiddenInputs) {
          const value = input.value;
          if (value && (value.includes('Bearer ') || value.length > 100)) {
            return value.replace('Bearer ', '');
          }
        }
        return null;
      });

      if (hiddenToken) {
        console.log('Token found in hidden input field');
        return hiddenToken;
      }

      return null;

    } catch (error) {
      console.error('Error extracting token:', error);
      return null;
    }
  }

  // Check if we're on the AgriStack domain and extract token directly
  async checkAgriStackToken(page) {
    try {
      const currentUrl = page.url();

      if (currentUrl.includes('mhfr.agristack.gov.in')) {
        console.log('On AgriStack domain - checking for token in localStorage...');

        const token = await page.evaluate(() => {
          return localStorage.getItem('token');
        });

        if (token) {
          console.log('✅ Token found on AgriStack domain');
          return { found: true, token: token, domain: 'agristack' };
        } else {
          console.log('❌ No token found on AgriStack domain');
          return { found: false, domain: 'agristack' };
        }
      }

      return { found: false, domain: 'other' };

    } catch (error) {
      console.error('Error checking AgriStack token:', error);
      return { found: false, error: error.message };
    }
  }

  // Check if the current page indicates a successful authentication
  async isPageAuthenticated(page) {
    try {
      const authStatus = await page.evaluate(() => {
        // Check for various authentication success indicators
        const indicators = [
          // Check if we're on a protected page
          !window.location.href.includes('login') && !window.location.href.includes('authorize'),

          // Check for user-specific content
          document.querySelector('[data-user]') || document.querySelector('.user-info'),

          // Check for logout buttons (indicates successful login)
          document.querySelector('[href*="logout"]') || document.querySelector('[onclick*="logout"]'),

          // Check for dashboard elements
          document.querySelector('.dashboard') || document.querySelector('[class*="dashboard"]'),

          // Check if we have any tokens in storage
          !!(localStorage.getItem('token') || localStorage.getItem('authToken') ||
            sessionStorage.getItem('token') || sessionStorage.getItem('authToken'))
        ];

        return {
          isAuthenticated: indicators.some(indicator => indicator),
          indicators: indicators,
          currentUrl: window.location.href,
          hasToken: !!(localStorage.getItem('token') || localStorage.getItem('authToken') ||
            sessionStorage.getItem('token') || sessionStorage.getItem('authToken'))
        };
      });

      console.log('Page authentication check:', authStatus);
      return authStatus;

    } catch (error) {
      console.error('Error checking page authentication status:', error);
      return { isAuthenticated: false, error: error.message };
    }
  }

  // Set token manually (for testing or manual updates)
  setToken(token, expiryHours = 12) {
    this.token = token;
    this.tokenExpiry = new Date(Date.now() + expiryHours * 60 * 60 * 1000);
    this.saveTokenToCache();
    console.log('Token manually set, expires at:', this.tokenExpiry);
  }
}

module.exports = TokenManager;
